/****************************************************************
#
# PART:         IDT70v7599
# MODULE:	IDT70v7599
# PURPOSE:      specific part
# DESIGNER:     Shawn M. Giguere
#
*****************************************************************
#
# HISTORY:
# 12/2/98 MFF  Initial version
# 2/17/99 MFF  Added CE to Read Error
# 3/23/99 MFF  Fixed memory access to require rising clock edge
# 7/27/99 MFF  reduced tccs_r to fix error
#
 *****************************************************************
 #   Copyright 1998, Integrated Device Technology, Inc.
 #   All Rights Reserved
 #
***************************************************************
* Integrated Device Technology, Inc. ("IDT") hereby grants the
 user of this Verilog/VCS model a non-exclusive, nontransferable 
 license to use this Verilog/VCS model under the following terms.   
 The user is granted this license only to use the Verilog/VCS    
 model and is not granted rights to sell, copy (except as needed   
 to run the IBIS model), rent, lease or sub-license the Verilog/VCS   
 model in whole or in part, or in modified form to anyone. The User   
 may modify the Verilog/VCS model to suit its specific applications,   
 but rights to derivative works and such modifications shall belong   
 to IDT. * 
*  This Verilog/VCS model is provided on an "AS IS" basis and
 IDT makes absolutely no warranty with respect to the information
 contained herein. IDT DISCLAIMS AND CUSTOMER WAIVES ALL
 WARRANTIES, EXPRESS AND IMPLIED, INCLUDING WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 ENTIRE RISK AS TO QUALITY AND PERFORMANCE IS WITH THE
 USER ACCORDINGLY, IN NO EVENT SHALL IDT BE LIABLE 
 FOR ANY DIRECT OR INDIRECT DAMAGES, WHETHER IN CONTRACT OR
 TORT, INCLUDING ANY LOST PROFITS OR OTHER INCIDENTAL,
 CONSEQUENTIAL, EXEMPLARY, OR PUNITIVE DAMAGES ARISING OUT OF
 THE USE OR APPLICATION OF THE VERILOG/VCS model.  Further, 
 IDT reserves the right to make changes without notice to any 
 product herein to improve reliability, function, or design.   
 IDT does not convey any license under patent rights or 
 any other intellectual property rights, including those of   
 third parties. IDT is not obligated to provide maintenance   
 or support for the licensed Verilog/VCS model. *
 ***************************************************************
#
# COMMENTS:
#
# 
#
****************************************************************/
/****************************************************************
#
# MODULE:	IDT70v7599
# PURPOSE:      hook together sub-modules
# DESIGNER:     Shawn Giguere
#
*****************************************************************
#
# HISTORY:
# 7/12/00 SMG	Initial version
#
****************************************************************/

`timescale 1ns / 100ps

// These switches define the part topology, change them for the different
// parts
// memory and array size
	`define width 36
	`define addr_bit 17
	`define depth 131072
// define topology
	`define x36

// change this line to select one of the speed grades below (five or six)
// Note tccs_r is an artifical spec to determine how long a read takes
// its based on cycle time more than anything

`define six


`ifdef seven.five
// performance specs
	 `define tsetup 1.8
	 `define thold 0.5
	 `define tccs 7.5
	 `define tcd 4.2
	 `define tcdft 12.0
	 `define tckhz 3.0
	 `define toe 4.2
	 `define tohz 3.7
 `endif

 `ifdef six
// performance specs
	 `define tsetup 1.7
	 `define thold 0.5
	 `define tccs 6.0
	 `define tcd 3.6
	 `define tcdft 10.0
	 `define tckhz 2.5
	 `define toe 3.6
	 `define tohz 3.5
 `endif

module idt70v7599(
	clkl, clkr,		// clock input
	rwxl, rwxr,		// read/write (low)
	ce0lx, ce1l,		// chip enables, 0 is active low, 1 is active high
	ce0rx, ce1r,		// chip enables, 0 is active low, 1 is active high
	belx, berx,		// byte enables, active low (2 or 4)
	oelx, oerx,		// output enable, active low
	l_a, r_a,  		// input address
	l_ba, r_ba,		// bank address
	adslx, adsrx, 		// address strobe
 	repeatlx, repeatrx, 	// counter resets
	cntenlx, cntenrx,	// counter enable
	iol, ior,		// ios
	optl, optr,		// options
	pipeftlx, pipeftrx,	// pipeline/flow through (low)
	tck,tdi,tms,trst,tdo	// jtag signals
           );

// IO declarations
input clkl, ce0lx, ce1l, oelx, rwxl, adslx, repeatlx, cntenlx, pipeftlx, optl;
input clkr, ce0rx, ce1r, oerx, rwxr, adsrx, repeatrx, cntenrx, pipeftrx, optr;
input	 [3:0]	 belx,berx;
input  [`addr_bit-7:0] l_a, r_a;
input  [5:0] l_ba, r_ba;
input tck, tdi, tms, trst;
inout [`width-1:0] iol, ior;
output tdo;

// internal registers
wire [`width-1:0] datal,datar;
wire [`addr_bit-1:0] al_i,ar_i;
wire [`addr_bit-1:0] al,ar;
wire [5:0] bal_i,bar_i;
wire [3:0] belx_i, berx_i;
wire cel_i, ce2l_i, rwlx_i, cer_i, ce2r_i, rwrx_i;

assign al = {l_ba[5:0],l_a[`addr_bit-7:0]};
assign ar = {r_ba[5:0],r_a[`addr_bit-7:0]};

io iol1(
	.ck(clkl),
	.rwx(rwxl),
	.ce0x(ce0lx),
	.ce1(ce1l),
	.bex(belx),
	.oex(oelx),
	.data(datal),
	.ce_buf(cel_i),
	.ce_2buf(ce2l_i),
	.rw_buf(rwlx_i),
	.be_buf(belx_i),
	.io(iol),
        .pfx(pipeftlx),
        .extest(extest)
	);

io ior1(
	.ck(clkr),
	.rwx(rwxr),
	.ce0x(ce0rx),
	.ce1(ce1r),
	.bex(berx),
	.oex(oerx),
	.data(datar),
	.ce_buf(cer_i),
	.ce_2buf(ce2r_i),
	.rw_buf(rwrx_i),
	.be_buf(berx_i),
	.io(ior),
        .pfx(pipeftrx),
        .extest(extest)
	);

count_reg lcounter(
	.clk(clkl),
	.bain(l_ba),
	.baout(bal_i),
	.ain(al),
	.aout(al_i),
	.repeatx(repeatlx),
	.adsx(adslx),
	.cntenx(cntenlx)
	);

count_reg rcounter(
	.clk(clkr),
	.bain(r_ba),
	.baout(bar_i),
	.ain(ar),
	.aout(ar_i),
	.repeatx(repeatrx),
	.adsx(adsrx),
	.cntenx(cntenrx)
	);

mem mcore(
	.clkl(clkl),
	.clkr(clkr),
	.bal(bal_i),
	.bar(bar_i),
	.al(al_i),
	.ar(ar_i),
	.rwxl(rwlx_i),
	.rwxr(rwrx_i),
	.cel(cel_i),
	.cer(cer_i),
	.ce2l(ce2l_i),
	.ce2r(ce2r_i),
	.datal(datal),
	.datar(datar),
	.bel(belx_i),
	.ber(berx_i),
	.pfxl(pipeftlx),
	.pfxr(pipeftrx)
	);	

jtag jtag_model(
	.lck_chip(clkl),
	.l_a_chip(l_a),
	.l_ba_chip(l_ba),
	.lio_chip(iol),
	.lrwx_chip(rwxl),
	.loex(oelx),
	.lbex_chip(belx),
	.lce0x_chip(ce0lx),
	.lce1_chip(ce1l),
	.lcntenx_chip(cntenlx),
	.ladsx_chip(adslx),
	.lopt_chip(optl),
	.lpfx_chip(pipeftlx),
	.lrepeatx_chip(repeatlx),
	.lio_jtag(iol),
	.rck_chip(clkr),
	.r_a_chip(r_a),
	.r_ba_chip(r_ba),
	.rio_chip(ior),
	.rrwx_chip(rwxr),
	.roex(oerx),
	.rbex_chip(berx),
	.rce0x_chip(ce0rx),
	.rce1_chip(ce1r),
	.rcntenx_chip(cntenrx),
	.radsx_chip(adsrx),
	.ropt_chip(optr),
	.rpfx_chip(pipeftrx),
	.rrepeatx_chip(repeatrx),
	.rio_jtag(ior),
	.tms(tms),
	.tck(tck),
	.tdi(tdi),
	.tdo(tdo),
	.trstx(trst),
        .extest(extest)
	);

endmodule

 /****************************************************************
#
# MODULE:	    io 
# PURPOSE:      data io and input control register
# DESIGNER:     Shawn Giguere
#
*****************************************************************
#
# HISTORY:
# 12/1/98 MFF  Initial version
#
*****************************************************************
#
# COMMENTS:
#
#  Output timing placed on control signals
#   and data register -- this makes the output assignment clean
#
****************************************************************/


module io(
	ck,		// clock input
	rwx, 		// read/write (low)
	ce0x, ce1,	// chip enables, 0 is active low, 1 is active high
	bex,		// byte enables, active low (2 or 4)
	oex,		// output enable, active low
	data,		// active high data to core
	ce_buf, ce_2buf,// ce to core (single ff output)
	rw_buf,		// rw to core (single ff output)
	be_buf,		// be to core (single ff output)
	io,		// pad ios				
        pfx,		// pipeline/flow through (low)
        extest          // extest disables chip outputs
           );


// error generation
reg [5:0] error;  //  [5]=ce1 [4]=rw [3:0]=be
reg derror;
reg flag_io, flag_ce, flag_rw, flag_be3, flag_be2;
reg flag_be1, flag_be0;


	 
// IO declarations
input	 [3:0]	 bex;
input ck, ce0x, ce1, oex, rwx, pfx, extest;
inout [`width-1:0]	data;
inout [`width-1:0]	io;
output ce_buf,ce_2buf, rw_buf;
output [3:0] be_buf;

// temporary registers
reg ce_2buf,ce_buf,rw_buf;
reg [3:0] be_ck,be_buf;
reg [`width-1:0] in_reg;
reg [`width-1:0] out_reg;
wire [3:0] en_io;
wire  oe_io, out_ck;



// generate error signals
specify
	$setuphold(posedge ck, io, `tsetup, `thold, flag_io);
	$setuphold(posedge ck, ce0x, `tsetup, `thold, flag_ce);
	$setuphold(posedge ck, ce1, `tsetup, `thold, flag_ce);
	$setuphold(posedge ck, rwx, `tsetup, `thold, flag_rw);
	$setuphold(posedge ck, bex[3], `tsetup, `thold, flag_be3);
	$setuphold(posedge ck, bex[2], `tsetup, `thold, flag_be2);
	$setuphold(posedge ck, bex[1], `tsetup, `thold, flag_be1);
	$setuphold(posedge ck, bex[0], `tsetup, `thold, flag_be0);
endspecify

always @(negedge ck or flag_io) derror <= (~ck) ? 1'b0 : 1'b1;
always @(negedge ck or flag_ce) error[5] <= (~ck) ? 1'b0 : 1'b1;
always @(negedge ck or flag_rw) error[4] <= (~ck) ? 1'b0 : 1'b1;
always @(negedge ck or flag_be3) error[3] <= (~ck) ? 1'b0 : 1'b1;
always @(negedge ck or flag_be2) error[2] <= (~ck) ? 1'b0 : 1'b1;
always @(negedge ck or flag_be1) error[1] <= (~ck) ? 1'b0 : 1'b1;
always @(negedge ck or flag_be0) error[0] <= (~ck) ? 1'b0 : 1'b1;


// output byte enables
assign #(`tcd, `tckhz) en_io[3] = rw_buf && ce_2buf && be_ck[3];
assign #(`tcd, `tckhz) en_io[2] = rw_buf && ce_2buf && be_ck[2];
assign #(`tcd, `tckhz) en_io[1] = rw_buf && ce_2buf && be_ck[1];
assign #(`tcd, `tckhz) en_io[0] = rw_buf && ce_2buf && be_ck[0];
assign #(`toe, `tohz) oe_io = ~oex;

assign #0.3 out_ck = ck;

assign io[8:0] = extest ? 9'bz : ((en_io[0]  && oe_io) ? out_reg[8:0] : 9'bz); 
`ifdef x18
	assign io[17:9] = extest ? 9'bz : ((en_io[1] && oe_io) ? out_reg[17:9] : 9'bz);
`endif
`ifdef x36 
	assign io[17:9] = extest ? 9'bz : ((en_io[1] && oe_io) ? out_reg[17:9] : 9'bz);
	assign io[26:18] = extest ? 9'bz : ((en_io[2] && oe_io) ? out_reg[26:18] : 9'bz); 
	assign io[35:27] = extest ? 9'bz : ((en_io[3] && oe_io) ? out_reg[35:27] : 9'bz); 
`endif


// setup data bus during writes
assign data[`width-1:0] = (~rw_buf) ? in_reg[`width-1:0] : `width'hz;

// 1st buffer and metastable state for controls
always @(posedge error[5] or posedge error[4] or posedge error[3] or 
			posedge error[2] or posedge error[1] or posedge error[0] or
			posedge ck)
begin
	ce_buf <=   (~error[5]) ? (~ce0x && ce1) : 1'bx;
	rw_buf <=   (~error[4]) ? rwx : 1'bx;
	be_buf[3] <=   (~error[3]) ? ~bex[3] : 1'bx;
	be_buf[2] <=   (~error[2]) ? ~bex[2] : 1'bx;
	be_buf[1] <=   (~error[1]) ? ~bex[1] : 1'bx;
	be_buf[0] <=   (~error[0]) ? ~bex[0] : 1'bx;
end

// double buffer
always @(posedge ck)
begin
	be_ck <= be_buf;
	ce_2buf <= ce_buf;
end
	
// input register
always @(posedge ck or posedge derror)
begin
	in_reg <= (derror) ? `width'hx : io;
end

// output register

always @(data)
  if (~pfx)
    out_reg <=  #(`tcdft-0.2) data;

always @(posedge out_ck)
  if (pfx)
    out_reg <= #(`tcd-0.3)  data;

endmodule

/****************************************************************
#
# MODULE:	memory core
# DESIGNER:     Shawn Giguere
#
*****************************************************************
#
# HISTORY:
# 11/30/98 MFF  Initial version
#
****************************************************************/


module mem(
	clkl, clkr,		// clocks
	al, ar,			// addresses
	bal, bar,		// addresses
	rwxl, rwxr,		// read/write(low)
	cel,cer,ce2l,ce2r,	// chip enable
	datal, datar,		// memory data to read or write
	bel, ber,		// byte enables (necessary for writes)
        pfxl, pfxr		// pipeline/flow through (low)
           );

// IO declarations
input  [5:0]  bar,bal;
input  [`addr_bit-1:0]  ar,al;
input	 [3:0]	 bel;
input	 [3:0]	 ber;
input  pfxl,pfxr,rwxl,rwxr,cel,cer,ce2l,ce2r,clkl,clkr;
inout [`width-1:0]	datal;
inout [`width-1:0]	datar;


// memory definition
	// each core represents 9 bits controlled by a single BE
reg [`width-1:0] mcore [`depth-1:0];

// temporary data registers
reg [`width-1:0] wbufl, wbufr, rdbufl, rdbufr, datal_reg, datar_reg;

// address valid registers
reg al_valid, ar_valid;

// error register
wire rwx_err;

// delayed clocks
wire mckl,mckr,mckl2,mckr2;

//read/write errors
assign rwx_err = (al_valid && ar_valid) ? (bal == bar) : 0;

// delayed clock is necessary to make sure that control changes
// on clock edge are recognized in memory operation
assign #0.1 mckl = clkl;  
assign #0.1 mckr = clkr;
assign #0.2 mckl2 = clkl;  
assign #0.2 mckr2 = clkr;

assign datal = (ce2l && rwxl) ? ((rwx_err) ? `width'hx : datal_reg) : `width'hz;
assign datar = (ce2r && rwxr) ? ((rwx_err) ? `width'hx : datar_reg) : `width'hz;

always @(posedge rwx_err) 
	$display("Read/Write Error: Both ports accessing same bank at time %0d",$time);
    
always @(posedge mckl)
        al_valid <= cel;
always @(negedge mckl)
	al_valid = 0;

always @(posedge mckr)
	ar_valid <= cer;
always @(negedge mckr)
	ar_valid = 0;

// l (Left) port access
always @(posedge mckl2)
 begin
   if ((cel || ce2l) && rwxl)
     begin
       if (pfxl)
         begin
           if (rwx_err) 
             begin
               datal_reg = rdbufl;             //put temp register on io
               rdbufl[`width-1:0] = `width'bx; //put data in temp register
             end
           else
             begin
               datal_reg = rdbufl;             //put temp register on io
               rdbufl[`width-1:0] = mcore[al]; //put data in temp register
             end
         end
       else
         begin
           if (rwx_err) 
             begin
               rdbufl[`width-1:0] = `width'bx; //put data in temp register
               datal_reg = `width'bx;          //put data on io
             end
           else
             begin
               rdbufl[`width-1:0] = mcore[al]; //put data in temp register
               datal_reg = mcore[al];          //put data on io
             end
         end
     end
   else
	if (cel && ~rwxl)
	  begin
                wbufl[`width-1:0] = mcore[al];
		if (rwx_err)
		  wbufl = `width'hx;
		else
		begin
			if ((bel[0]) && (rwx_err == 0)) wbufl[8:0]=datal[8:0];
			else if (bel[0] !== 0) wbufl[8:0] = 9'bx;
			`ifdef x18  // only use these lines if data is 18 bits
				if ((bel[1]) && (rwx_err == 0)) wbufl[17:9]=datal[17:9];
				else if (bel[1] !== 0) wbufl[17:9] = 9'bx;
			`endif
			`ifdef x36  // only use these lines if data is 36 bits
				if ((bel[1]) && (rwx_err == 0)) wbufl[17:9]=datal[17:9];
				else if (bel[1] !== 0) wbufl[17:9] = 9'bx;
				if ((bel[2]) && (rwx_err == 0)) wbufl[26:18]=datal[26:18];
				else if (bel[2] !== 0) wbufl[26:18] = 9'bx;
				if ((bel[3]) && (rwx_err == 0)) wbufl[35:27]=datal[35:27];
				else if (bel[3] !== 0) wbufl[35:27] = 9'bx;
			`endif
		end // end else
		mcore[al] = wbufl[`width-1:0];
	end // if ce && ~rwxl
 end	


// r (right) port access
always @(posedge mckr2)
 begin
   if ((cer || ce2r) && rwxr)
     begin
       if (pfxr)
         begin
           if (rwx_err) 
             begin
               datar_reg = rdbufr;             //put temp register on io
               rdbufr[`width-1:0] = `width'bx; //put data in temp register
             end
           else
             begin
               datar_reg = rdbufr;             //put temp register on io
               rdbufr[`width-1:0] = mcore[ar]; //put data in temp register
             end
         end
       else
         begin
           if (rwx_err) 
             begin
               rdbufr[`width-1:0] = `width'bx; //put data in temp register
               datar_reg = `width'bx;          //put data on io
             end
           else
             begin
               rdbufr[`width-1:0] = mcore[ar]; //put data in temp register
               datar_reg = mcore[ar];          //put data on io
             end
         end
     end
   else
	if (cer && ~rwxr)
	begin
                wbufr[`width-1:0] = mcore[ar];
		if (rwx_err)
			wbufr = `width'hx;
		else
		begin
			if ((ber[0]) && (rwx_err == 0)) wbufr[8:0]=datar[8:0];
			else if (ber[0] !== 0) wbufr[8:0] = 9'bx;
			`ifdef x18  // only use these lines if data is 18 bits
			  if ((ber[1]) && (rwx_err == 0)) wbufr[17:9]=datar[17:9];
			  else if (ber[1] !== 0) wbufr[17:9] = 9'bx;
			`endif
			`ifdef x36  // only use these lines if data is 36 bits
				if ((ber[1]) && (rwx_err == 0)) wbufr[17:9]=datar[17:9];
				else if (ber[1] !== 0) wbufr[17:9] = 9'bx;
				if ((ber[2]) && (rwx_err == 0)) wbufr[26:18]=datar[26:18];
				else if (ber[2] !== 0) wbufr[26:18] = 9'bx;
				if ((ber[3]) && (rwx_err == 0)) wbufr[35:27]=datar[35:27];
				else if (ber[3] !== 0) wbufr[35:27] = 9'bx;
			`endif
		end // end else
	  mcore[ar] = wbufr[`width-1:0];
	end // if ce && ~rwxr
 end	


endmodule

/****************************************************************
#
# MODULE:	counter/address register
# PURPOSE:      A 14 to 16 bit counter (depending on the part)
#			and address register
# DESIGNER:     Shawn Giguere
#
*****************************************************************
#
# HISTORY:
# 11/30/98 MFF  Initial version
#
****************************************************************/

module count_reg (clk, ain, bain, aout, baout, repeatx, adsx, cntenx);

// IO declarations
input  [`addr_bit-1:0]	ain;
input  [5:0]		bain;
input	clk,repeatx,adsx,cntenx;
output [`addr_bit-1:0]	aout;
output [5:0]		baout;

// output registers
reg [`addr_bit-1:0]	aout;
reg [5:0]		baout;
reg [`addr_bit-1:0]	repeat_adr;
reg [5:0]		repeat_badr;

// error generation
reg flag, error;
specify
	$setuphold(posedge clk, ain, `tsetup, `thold, flag);
	$setuphold(posedge clk, repeatx, `tsetup, `thold, flag);
	$setuphold(posedge clk, adsx, `tsetup, `thold, flag);
	$setuphold(posedge clk, cntenx,	`tsetup, `thold, flag);
endspecify

always @(negedge clk or flag) error <= (~clk) ? 1'b0 : 1'b1;

always @(posedge clk or posedge error)
	begin
	if (adsx)
		begin
		repeat_adr <= ain;
		repeat_badr <= bain;
		end
	if (error)
		begin
		aout <= `addr_bit'hx;
		baout <= 6'hx;
		end
	else
		begin
		aout <= (repeatx) ? ((adsx) ? ((cntenx) ? aout: aout+1) : ain) : repeat_adr;
		baout <= (repeatx) ? ((adsx) ? ((cntenx) ? baout: baout+1) : bain) : repeat_badr;
		end
 	end

endmodule

/*****************************************************************************
#
# MODULE: 	jtag
# PURPOSE: 	jtag boundary scan chain
# DESIGNER:	Chenhao Geng
# MODIFIED:	Shawn M. Giguere
#
*****************************************************************************/

module jtag (lck_chip, l_a_chip, l_ba_chip, lio_chip, lrwx_chip, loex, lbex_chip, lopt_chip,
	     lce0x_chip, lce1_chip, ladsx_chip, lcntenx_chip, lpfx_chip, lrepeatx_chip, lio_jtag,
	     rck_chip, r_a_chip, r_ba_chip, rio_chip, rrwx_chip, roex, rbex_chip, ropt_chip,
	     rce0x_chip, rce1_chip, radsx_chip, rcntenx_chip, rpfx_chip, rrepeatx_chip, rio_jtag,
	     tms, tck, tdi, tdo, trstx, extest);
input	[`addr_bit-7:0] l_a_chip,r_a_chip;
input	[5:0] l_ba_chip,r_ba_chip;
input	[`width-1:0] lio_chip,rio_chip;
output	[`width-1:0] lio_jtag,rio_jtag;
input	[3:0]	lbex_chip,rbex_chip;
input		lrwx_chip,loex,lce0x_chip,lce1_chip,lpfx_chip,lrepeatx_chip;
input		lck_chip, lopt_chip, ladsx_chip, lcntenx_chip;
input		rrwx_chip,roex,rce0x_chip,rce1_chip,rpfx_chip,rrepeatx_chip;
input		rck_chip, ropt_chip, radsx_chip, rcntenx_chip;
input		tms, tck, tdi, trstx;
output		tdo;
inout		extest;
wire		shiftdr, updatedr, clockdr, highz, jtoe, tdo_tap, trst;

wire	[207:0]	so,do;

assign tdo = jtoe ? tdo_tap : 1'bz;
//assign tdo = highz ? 1'bz : (jtoe ? tdo_tap : 1'bz);
assign trst = ~trstx;

idt70v7599_tap tap0 (tck, trst, tms, tdi, tdo_tap, jtoe, so[0], tdi, shiftdr, updatedr, clockdr, extest, highz);

idt70v7599_incell bscell_205(tck, tdi, pipeftl_chip, shiftdr, clockdr, so[205]);
idt70v7599_incell bscell_204(tck, so[205], lio_chip[18], shiftdr, clockdr, so[204]);
idt70v7599_iocell bscell_203(tck, so[204], lio_chip[18], do[17], shiftdr, clockdr, updatedr, extest, highz, so[203], lio_jtag[18], trst);
idt70v7599_incell bscell_202(tck, so[203], rio_chip[18], shiftdr, clockdr, so[202]);
idt70v7599_iocell bscell_201(tck, so[202], rio_chip[18], do[114], shiftdr, clockdr, updatedr, extest, highz, so[201], rio_jtag[18], trst);
idt70v7599_incell bscell_200(tck, so[201], lio_chip[19], shiftdr, clockdr, so[200]);
idt70v7599_iocell bscell_199(tck, so[200], lio_chip[19], do[17], shiftdr, clockdr, updatedr, extest, highz, so[199], lio_jtag[19], trst);
idt70v7599_incell bscell_198(tck, so[199], rio_chip[19], shiftdr, clockdr, so[198]);
idt70v7599_iocell bscell_197(tck, so[198], rio_chip[19], do[114], shiftdr, clockdr, updatedr, extest, highz, so[197], rio_jtag[19], trst);
idt70v7599_incell bscell_196(tck, so[197], lio_chip[20], shiftdr, clockdr, so[196]);
idt70v7599_iocell bscell_195(tck, so[196], lio_chip[20], do[17], shiftdr, clockdr, updatedr, extest, highz, so[195], lio_jtag[20], trst);
idt70v7599_incell bscell_194(tck, so[195], rio_chip[20], shiftdr, clockdr, so[194]);
idt70v7599_iocell bscell_193(tck, so[194], rio_chip[20], do[114], shiftdr, clockdr, updatedr, extest, highz, so[193], rio_jtag[20], trst);
idt70v7599_incell bscell_192(tck, so[193], lio_chip[21], shiftdr, clockdr, so[192]);
idt70v7599_iocell bscell_191(tck, so[192], lio_chip[21], do[17], shiftdr, clockdr, updatedr, extest, highz, so[191], lio_jtag[21], trst);
idt70v7599_incell bscell_190(tck, so[191], rio_chip[21], shiftdr, clockdr, so[190]);
idt70v7599_iocell bscell_189(tck, so[190], rio_chip[21], do[114], shiftdr, clockdr, updatedr, extest, highz, so[189], rio_jtag[21], trst);
idt70v7599_incell bscell_188(tck, so[189], lio_chip[22], shiftdr, clockdr, so[188]);
idt70v7599_iocell bscell_187(tck, so[188], lio_chip[22], do[17], shiftdr, clockdr, updatedr, extest, highz, so[187], lio_jtag[22], trst);
idt70v7599_incell bscell_186(tck, so[187], rio_chip[22], shiftdr, clockdr, so[186]);
idt70v7599_iocell bscell_185(tck, so[186], rio_chip[22], do[114], shiftdr, clockdr, updatedr, extest, highz, so[185], rio_jtag[22], trst);
idt70v7599_incell bscell_184(tck, so[185], lio_chip[23], shiftdr, clockdr, so[184]);
idt70v7599_iocell bscell_183(tck, so[184], lio_chip[23], do[17], shiftdr, clockdr, updatedr, extest, highz, so[183], lio_jtag[23], trst);
idt70v7599_incell bscell_182(tck, so[183], rio_chip[23], shiftdr, clockdr, so[182]);
idt70v7599_iocell bscell_181(tck, so[182], rio_chip[23], do[114], shiftdr, clockdr, updatedr, extest, highz, so[181], rio_jtag[23], trst);
idt70v7599_incell bscell_180(tck, so[181], lio_chip[24], shiftdr, clockdr, so[180]);
idt70v7599_iocell bscell_179(tck, so[180], lio_chip[24], do[17], shiftdr, clockdr, updatedr, extest, highz, so[179], lio_jtag[24], trst);
idt70v7599_incell bscell_178(tck, so[179], rio_chip[24], shiftdr, clockdr, so[178]);
idt70v7599_iocell bscell_177(tck, so[178], rio_chip[24], do[114], shiftdr, clockdr, updatedr, extest, highz, so[177], rio_jtag[24], trst);
idt70v7599_incell bscell_176(tck, so[177], lio_chip[25], shiftdr, clockdr, so[176]);
idt70v7599_iocell bscell_175(tck, so[176], lio_chip[25], do[17], shiftdr, clockdr, updatedr, extest, highz, so[175], lio_jtag[25], trst);
idt70v7599_incell bscell_174(tck, so[175], rio_chip[25], shiftdr, clockdr, so[174]);
idt70v7599_iocell bscell_173(tck, so[174], rio_chip[25], do[114], shiftdr, clockdr, updatedr, extest, highz, so[173], rio_jtag[25], trst);
idt70v7599_incell bscell_172(tck, so[173], lio_chip[26], shiftdr, clockdr, so[172]);
idt70v7599_iocell bscell_171(tck, so[172], lio_chip[26], do[17], shiftdr, clockdr, updatedr, extest, highz, so[171], lio_jtag[26], trst);
idt70v7599_incell bscell_170(tck, so[171], rio_chip[26], shiftdr, clockdr, so[170]);
idt70v7599_iocell bscell_169(tck, so[170], rio_chip[26], do[114], shiftdr, clockdr, updatedr, extest, highz, so[169], rio_jtag[26], trst);
idt70v7599_incell bscell_168(tck, so[169], rio_chip[27], shiftdr, clockdr, so[168]);
idt70v7599_iocell bscell_167(tck, so[168], rio_chip[27], do[114], shiftdr, clockdr, updatedr, extest, highz, so[167], rio_jtag[27], trst);
idt70v7599_incell bscell_166(tck, so[167], lio_chip[27], shiftdr, clockdr, so[166]);
idt70v7599_iocell bscell_165(tck, so[166], lio_chip[27], do[17], shiftdr, clockdr, updatedr, extest, highz, so[165], lio_jtag[27], trst);
idt70v7599_incell bscell_164(tck, so[165], rio_chip[28], shiftdr, clockdr, so[164]);
idt70v7599_iocell bscell_163(tck, so[164], rio_chip[28], do[114], shiftdr, clockdr, updatedr, extest, highz, so[163], rio_jtag[28], trst);
idt70v7599_incell bscell_162(tck, so[163], lio_chip[28], shiftdr, clockdr, so[162]);
idt70v7599_iocell bscell_161(tck, so[162], lio_chip[28], do[17], shiftdr, clockdr, updatedr, extest, highz, so[161], lio_jtag[28], trst);
idt70v7599_incell bscell_160(tck, so[161], rio_chip[29], shiftdr, clockdr, so[160]);
idt70v7599_iocell bscell_159(tck, so[160], rio_chip[29], do[114], shiftdr, clockdr, updatedr, extest, highz, so[159], rio_jtag[29], trst);
idt70v7599_incell bscell_158(tck, so[159], lio_chip[29], shiftdr, clockdr, so[158]);
idt70v7599_iocell bscell_157(tck, so[158], lio_chip[29], do[17], shiftdr, clockdr, updatedr, extest, highz, so[157], lio_jtag[29], trst);
idt70v7599_incell bscell_156(tck, so[157], rio_chip[30], shiftdr, clockdr, so[156]);
idt70v7599_iocell bscell_155(tck, so[156], rio_chip[30], do[114], shiftdr, clockdr, updatedr, extest, highz, so[155], rio_jtag[30], trst);
idt70v7599_incell bscell_154(tck, so[155], lio_chip[30], shiftdr, clockdr, so[154]);
idt70v7599_iocell bscell_153(tck, so[154], lio_chip[30], do[17], shiftdr, clockdr, updatedr, extest, highz, so[153], lio_jtag[30], trst);
idt70v7599_incell bscell_152(tck, so[153], rio_chip[31], shiftdr, clockdr, so[152]);
idt70v7599_iocell bscell_151(tck, so[152], rio_chip[31], do[114], shiftdr, clockdr, updatedr, extest, highz, so[151], rio_jtag[31], trst);
idt70v7599_incell bscell_150(tck, so[151], lio_chip[31], shiftdr, clockdr, so[150]);
idt70v7599_iocell bscell_149(tck, so[150], lio_chip[31], do[17], shiftdr, clockdr, updatedr, extest, highz, so[149], lio_jtag[31], trst);
idt70v7599_incell bscell_148(tck, so[149], rio_chip[32], shiftdr, clockdr, so[148]);
idt70v7599_iocell bscell_147(tck, so[148], rio_chip[32], do[114], shiftdr, clockdr, updatedr, extest, highz, so[147], rio_jtag[32], trst);
idt70v7599_incell bscell_146(tck, so[147], lio_chip[32], shiftdr, clockdr, so[146]);
idt70v7599_iocell bscell_145(tck, so[146], lio_chip[32], do[17], shiftdr, clockdr, updatedr, extest, highz, so[145], lio_jtag[32], trst);
idt70v7599_incell bscell_144(tck, so[145], rio_chip[33], shiftdr, clockdr, so[144]);
idt70v7599_iocell bscell_143(tck, so[144], rio_chip[33], do[114], shiftdr, clockdr, updatedr, extest, highz, so[143], rio_jtag[33], trst);
idt70v7599_incell bscell_142(tck, so[143], lio_chip[33], shiftdr, clockdr, so[142]);
idt70v7599_iocell bscell_141(tck, so[142], lio_chip[33], do[17], shiftdr, clockdr, updatedr, extest, highz, so[141], lio_jtag[33], trst);
idt70v7599_incell bscell_140(tck, so[141], rio_chip[34], shiftdr, clockdr, so[140]);
idt70v7599_iocell bscell_139(tck, so[140], rio_chip[34], do[114], shiftdr, clockdr, updatedr, extest, highz, so[139], rio_jtag[34], trst);
idt70v7599_incell bscell_138(tck, so[139], lio_chip[34], shiftdr, clockdr, so[138]);
idt70v7599_iocell bscell_137(tck, so[138], lio_chip[34], do[17], shiftdr, clockdr, updatedr, extest, highz, so[137], lio_jtag[34], trst);
idt70v7599_incell bscell_136(tck, so[137], rio_chip[35], shiftdr, clockdr, so[136]);
idt70v7599_iocell bscell_135(tck, so[136], rio_chip[35], do[114], shiftdr, clockdr, updatedr, extest, highz, so[135], rio_jtag[35], trst);
idt70v7599_incell bscell_134(tck, so[135], lio_chip[35], shiftdr, clockdr, so[134]);
idt70v7599_iocell bscell_133(tck, so[134], lio_chip[35], do[17], shiftdr, clockdr, updatedr, extest, highz, so[133], lio_jtag[35], trst);
idt70v7599_incell bscell_132(tck, so[133], pipeftr_chip, shiftdr, clockdr, so[132]);
idt70v7599_incell bscell_131(tck, so[132], r_ba_chip[5], shiftdr, clockdr, so[131]);
idt70v7599_incell bscell_130(tck, so[131], r_ba_chip[4], shiftdr, clockdr, so[130]);
idt70v7599_incell bscell_129(tck, so[130], r_ba_chip[3], shiftdr, clockdr, so[129]);
idt70v7599_incell bscell_128(tck, so[129], r_ba_chip[2], shiftdr, clockdr, so[128]);
idt70v7599_incell bscell_127(tck, so[128], r_ba_chip[1], shiftdr, clockdr, so[127]);
idt70v7599_incell bscell_126(tck, so[127], r_ba_chip[0], shiftdr, clockdr, so[126]);
idt70v7599_incell bscell_125(tck, so[126], r_a_chip[10], shiftdr, clockdr, so[125]);
idt70v7599_incell bscell_124(tck, so[125], r_a_chip[9], shiftdr, clockdr, so[124]);
idt70v7599_incell bscell_123(tck, so[124], r_a_chip[8], shiftdr, clockdr, so[123]);
idt70v7599_incell bscell_122(tck, so[123], r_a_chip[7], shiftdr, clockdr, so[122]);
idt70v7599_incell bscell_121(tck, so[122], be3r_chip, shiftdr, clockdr, so[121]);
idt70v7599_incell bscell_120(tck, so[121], be2r_chip, shiftdr, clockdr, so[120]);
idt70v7599_incell bscell_119(tck, so[120], be1r_chip, shiftdr, clockdr, so[119]);
idt70v7599_incell bscell_118(tck, so[119], be0r_chip, shiftdr, clockdr, so[118]);
idt70v7599_incell bscell_117(tck, so[118], ce1r_chip, shiftdr, clockdr, so[117]);
idt70v7599_incell bscell_116(tck, so[117], ce0r_chip, shiftdr, clockdr, so[116]);
idt70v7599_incell bscell_115(tck, so[116], clkr_chip, shiftdr, clockdr, so[115]);
idt70v7599_oecell bscell_114(tck, so[115], roex, shiftdr, clockdr, updatedr, extest, so[114], do[114]);
idt70v7599_incell bscell_113(tck, so[114], rwr_chip, shiftdr, clockdr, so[113]);
idt70v7599_incell bscell_112(tck, so[113], adsr_chip, shiftdr, clockdr, so[112]);
idt70v7599_incell bscell_111(tck, so[112], cntenr_chip, shiftdr, clockdr, so[111]);
idt70v7599_incell bscell_110(tck, so[111], repeatr_chip, shiftdr, clockdr, so[110]);
idt70v7599_incell bscell_109(tck, so[110], r_a_chip[6], shiftdr, clockdr, so[109]);
idt70v7599_incell bscell_108(tck, so[109], r_a_chip[5], shiftdr, clockdr, so[108]);
idt70v7599_incell bscell_107(tck, so[108], r_a_chip[4], shiftdr, clockdr, so[107]);
idt70v7599_incell bscell_106(tck, so[107], r_a_chip[3], shiftdr, clockdr, so[106]);
idt70v7599_incell bscell_105(tck, so[106], r_a_chip[2], shiftdr, clockdr, so[105]);
idt70v7599_incell bscell_104(tck, so[105], r_a_chip[1], shiftdr, clockdr, so[104]);
idt70v7599_incell bscell_103(tck, so[104], r_a_chip[0], shiftdr, clockdr, so[103]);
idt70v7599_incell bscell_102(tck, so[103], optr_chip, shiftdr, clockdr, so[102]);
idt70v7599_incell bscell_101(tck, so[102], lio_chip[0], shiftdr, clockdr, so[101]);
idt70v7599_iocell bscell_100(tck, so[101], lio_chip[0], do[17], shiftdr, clockdr, updatedr, extest, highz, so[100], lio_jtag[0], trst);
idt70v7599_incell bscell_99(tck, so[100], rio_chip[0], shiftdr, clockdr, so[99]);
idt70v7599_iocell bscell_98(tck, so[99], rio_chip[0], do[114], shiftdr, clockdr, updatedr, extest, highz, so[98], rio_jtag[0], trst);
idt70v7599_incell bscell_97(tck, so[98], lio_chip[1], shiftdr, clockdr, so[97]);
idt70v7599_iocell bscell_96(tck, so[97], lio_chip[1], do[17], shiftdr, clockdr, updatedr, extest, highz, so[96], lio_jtag[1], trst);
idt70v7599_incell bscell_95(tck, so[96], rio_chip[1], shiftdr, clockdr, so[95]);
idt70v7599_iocell bscell_94(tck, so[95], rio_chip[1], do[114], shiftdr, clockdr, updatedr, extest, highz, so[94], rio_jtag[1], trst);
idt70v7599_incell bscell_93(tck, so[94], lio_chip[2], shiftdr, clockdr, so[93]);
idt70v7599_iocell bscell_92(tck, so[93], lio_chip[2], do[17], shiftdr, clockdr, updatedr, extest, highz, so[92], lio_jtag[2], trst);
idt70v7599_incell bscell_91(tck, so[92], rio_chip[2], shiftdr, clockdr, so[91]);
idt70v7599_iocell bscell_90(tck, so[91], rio_chip[2], do[114], shiftdr, clockdr, updatedr, extest, highz, so[90], rio_jtag[2], trst);
idt70v7599_incell bscell_89(tck, so[90], lio_chip[3], shiftdr, clockdr, so[89]);
idt70v7599_iocell bscell_88(tck, so[89], lio_chip[3], do[17], shiftdr, clockdr, updatedr, extest, highz, so[88], lio_jtag[3], trst);
idt70v7599_incell bscell_87(tck, so[88], rio_chip[3], shiftdr, clockdr, so[87]);
idt70v7599_iocell bscell_86(tck, so[87], rio_chip[3], do[114], shiftdr, clockdr, updatedr, extest, highz, so[86], rio_jtag[3], trst);
idt70v7599_incell bscell_85(tck, so[86], lio_chip[4], shiftdr, clockdr, so[85]);
idt70v7599_iocell bscell_84(tck, so[85], lio_chip[4], do[17], shiftdr, clockdr, updatedr, extest, highz, so[84], lio_jtag[4], trst);
idt70v7599_incell bscell_83(tck, so[84], rio_chip[4], shiftdr, clockdr, so[83]);
idt70v7599_iocell bscell_82(tck, so[83], rio_chip[4], do[114], shiftdr, clockdr, updatedr, extest, highz, so[82], rio_jtag[4], trst);
idt70v7599_incell bscell_81(tck, so[82], lio_chip[5], shiftdr, clockdr, so[81]);
idt70v7599_iocell bscell_80(tck, so[81], lio_chip[5], do[17], shiftdr, clockdr, updatedr, extest, highz, so[80], lio_jtag[5], trst);
idt70v7599_incell bscell_79(tck, so[80], rio_chip[5], shiftdr, clockdr, so[79]);
idt70v7599_iocell bscell_78(tck, so[79], rio_chip[5], do[114], shiftdr, clockdr, updatedr, extest, highz, so[78], rio_jtag[5], trst);
idt70v7599_incell bscell_77(tck, so[78], lio_chip[6], shiftdr, clockdr, so[77]);
idt70v7599_iocell bscell_76(tck, so[77], lio_chip[6], do[17], shiftdr, clockdr, updatedr, extest, highz, so[76], lio_jtag[6], trst);
idt70v7599_incell bscell_75(tck, so[76], rio_chip[6], shiftdr, clockdr, so[75]);
idt70v7599_iocell bscell_74(tck, so[75], rio_chip[6], do[114], shiftdr, clockdr, updatedr, extest, highz, so[74], rio_jtag[6], trst);
idt70v7599_incell bscell_73(tck, so[74], lio_chip[7], shiftdr, clockdr, so[73]);
idt70v7599_iocell bscell_72(tck, so[73], lio_chip[7], do[17], shiftdr, clockdr, updatedr, extest, highz, so[72], lio_jtag[7], trst);
idt70v7599_incell bscell_71(tck, so[72], rio_chip[7], shiftdr, clockdr, so[71]);
idt70v7599_iocell bscell_70(tck, so[71], rio_chip[7], do[114], shiftdr, clockdr, updatedr, extest, highz, so[70], rio_jtag[7], trst);
idt70v7599_incell bscell_69(tck, so[70], lio_chip[8], shiftdr, clockdr, so[69]);
idt70v7599_iocell bscell_68(tck, so[69], lio_chip[8], do[17], shiftdr, clockdr, updatedr, extest, highz, so[68], lio_jtag[8], trst);
idt70v7599_incell bscell_67(tck, so[68], rio_chip[8], shiftdr, clockdr, so[67]);
idt70v7599_iocell bscell_66(tck, so[67], rio_chip[8], do[114], shiftdr, clockdr, updatedr, extest, highz, so[66], rio_jtag[8], trst);
idt70v7599_incell bscell_65(tck, so[66], rio_chip[9], shiftdr, clockdr, so[65]);
idt70v7599_iocell bscell_64(tck, so[65], rio_chip[9], do[114], shiftdr, clockdr, updatedr, extest, highz, so[64], rio_jtag[9], trst);
idt70v7599_incell bscell_63(tck, so[64], lio_chip[9], shiftdr, clockdr, so[63]);
idt70v7599_iocell bscell_62(tck, so[63], lio_chip[9], do[17], shiftdr, clockdr, updatedr, extest, highz, so[62], lio_jtag[9], trst);
idt70v7599_incell bscell_61(tck, so[62], rio_chip[10], shiftdr, clockdr, so[61]);
idt70v7599_iocell bscell_60(tck, so[61], rio_chip[10], do[114], shiftdr, clockdr, updatedr, extest, highz, so[60], rio_jtag[10], trst);
idt70v7599_incell bscell_59(tck, so[60], lio_chip[10], shiftdr, clockdr, so[59]);
idt70v7599_iocell bscell_58(tck, so[59], lio_chip[10], do[17], shiftdr, clockdr, updatedr, extest, highz, so[58], lio_jtag[10], trst);
idt70v7599_incell bscell_57(tck, so[58], rio_chip[11], shiftdr, clockdr, so[57]);
idt70v7599_iocell bscell_56(tck, so[57], rio_chip[11], do[114], shiftdr, clockdr, updatedr, extest, highz, so[56], rio_jtag[11], trst);
idt70v7599_incell bscell_55(tck, so[56], lio_chip[11], shiftdr, clockdr, so[55]);
idt70v7599_iocell bscell_54(tck, so[55], lio_chip[11], do[17], shiftdr, clockdr, updatedr, extest, highz, so[54], lio_jtag[11], trst);
idt70v7599_incell bscell_53(tck, so[54], rio_chip[12], shiftdr, clockdr, so[53]);
idt70v7599_iocell bscell_52(tck, so[53], rio_chip[12], do[114], shiftdr, clockdr, updatedr, extest, highz, so[52], rio_jtag[12], trst);
idt70v7599_incell bscell_51(tck, so[52], lio_chip[12], shiftdr, clockdr, so[51]);
idt70v7599_iocell bscell_50(tck, so[51], lio_chip[12], do[17], shiftdr, clockdr, updatedr, extest, highz, so[50], lio_jtag[12], trst);
idt70v7599_incell bscell_49(tck, so[50], rio_chip[13], shiftdr, clockdr, so[49]);
idt70v7599_iocell bscell_48(tck, so[49], rio_chip[13], do[114], shiftdr, clockdr, updatedr, extest, highz, so[48], rio_jtag[13], trst);
idt70v7599_incell bscell_47(tck, so[48], lio_chip[13], shiftdr, clockdr, so[47]);
idt70v7599_iocell bscell_46(tck, so[47], lio_chip[13], do[17], shiftdr, clockdr, updatedr, extest, highz, so[46], lio_jtag[13], trst);
idt70v7599_incell bscell_45(tck, so[46], rio_chip[14], shiftdr, clockdr, so[45]);
idt70v7599_iocell bscell_44(tck, so[45], rio_chip[14], do[114], shiftdr, clockdr, updatedr, extest, highz, so[44], rio_jtag[14], trst);
idt70v7599_incell bscell_43(tck, so[44], lio_chip[14], shiftdr, clockdr, so[43]);
idt70v7599_iocell bscell_42(tck, so[43], lio_chip[14], do[17], shiftdr, clockdr, updatedr, extest, highz, so[42], lio_jtag[14], trst);
idt70v7599_incell bscell_41(tck, so[42], rio_chip[15], shiftdr, clockdr, so[41]);
idt70v7599_iocell bscell_40(tck, so[41], rio_chip[15], do[114], shiftdr, clockdr, updatedr, extest, highz, so[40], rio_jtag[15], trst);
idt70v7599_incell bscell_39(tck, so[40], lio_chip[15], shiftdr, clockdr, so[39]);
idt70v7599_iocell bscell_38(tck, so[39], lio_chip[15], do[17], shiftdr, clockdr, updatedr, extest, highz, so[38], lio_jtag[15], trst);
idt70v7599_incell bscell_37(tck, so[38], rio_chip[16], shiftdr, clockdr, so[37]);
idt70v7599_iocell bscell_36(tck, so[37], rio_chip[16], do[114], shiftdr, clockdr, updatedr, extest, highz, so[36], rio_jtag[16], trst);
idt70v7599_incell bscell_35(tck, so[36], lio_chip[16], shiftdr, clockdr, so[35]);
idt70v7599_iocell bscell_34(tck, so[35], lio_chip[16], do[17], shiftdr, clockdr, updatedr, extest, highz, so[34], lio_jtag[16], trst);
idt70v7599_incell bscell_33(tck, so[34], rio_chip[17], shiftdr, clockdr, so[33]);
idt70v7599_iocell bscell_32(tck, so[33], rio_chip[17], do[114], shiftdr, clockdr, updatedr, extest, highz, so[32], rio_jtag[17], trst);
idt70v7599_incell bscell_31(tck, so[32], lio_chip[17], shiftdr, clockdr, so[31]);
idt70v7599_iocell bscell_30(tck, so[31], lio_chip[17], do[17], shiftdr, clockdr, updatedr, extest, highz, so[30], lio_jtag[17], trst);
idt70v7599_incell bscell_29(tck, so[30], optl_chip, shiftdr, clockdr, so[29]);
idt70v7599_incell bscell_28(tck, so[29], l_a_chip[0], shiftdr, clockdr, so[28]);
idt70v7599_incell bscell_27(tck, so[28], l_a_chip[1], shiftdr, clockdr, so[27]);
idt70v7599_incell bscell_26(tck, so[27], l_a_chip[2], shiftdr, clockdr, so[26]);
idt70v7599_incell bscell_25(tck, so[26], l_a_chip[3], shiftdr, clockdr, so[25]);
idt70v7599_incell bscell_24(tck, so[25], l_a_chip[4], shiftdr, clockdr, so[24]);
idt70v7599_incell bscell_23(tck, so[24], l_a_chip[5], shiftdr, clockdr, so[23]);
idt70v7599_incell bscell_22(tck, so[23], l_a_chip[6], shiftdr, clockdr, so[22]);
idt70v7599_incell bscell_21(tck, so[22], repeatl_chip, shiftdr, clockdr, so[21]);
idt70v7599_incell bscell_20(tck, so[21], cntenl_chip, shiftdr, clockdr, so[20]);
idt70v7599_incell bscell_19(tck, so[20], adsl_chip, shiftdr, clockdr, so[19]);
idt70v7599_incell bscell_18(tck, so[19], rwl_chip, shiftdr, clockdr, so[18]);
idt70v7599_oecell bscell_17(tck, so[18], loex, shiftdr, clockdr, updatedr, extest, so[17], do[17]);
idt70v7599_incell bscell_16(tck, so[17], clkl_chip, shiftdr, clockdr, so[16]);
idt70v7599_incell bscell_15(tck, so[16], ce0l_chip, shiftdr, clockdr, so[15]);
idt70v7599_incell bscell_14(tck, so[15], ce1l_chip, shiftdr, clockdr, so[14]);
idt70v7599_incell bscell_13(tck, so[14], be0l_chip, shiftdr, clockdr, so[13]);
idt70v7599_incell bscell_12(tck, so[13], be1l_chip, shiftdr, clockdr, so[12]);
idt70v7599_incell bscell_11(tck, so[12], be2l_chip, shiftdr, clockdr, so[11]);
idt70v7599_incell bscell_10(tck, so[11], be3l_chip, shiftdr, clockdr, so[10]);
idt70v7599_incell bscell_9(tck, so[10], l_a_chip[7], shiftdr, clockdr, so[9]);
idt70v7599_incell bscell_8(tck, so[9], l_a_chip[8], shiftdr, clockdr, so[8]);
idt70v7599_incell bscell_7(tck, so[8], l_a_chip[9], shiftdr, clockdr, so[7]);
idt70v7599_incell bscell_6(tck, so[7], l_a_chip[10], shiftdr, clockdr, so[6]);
idt70v7599_incell bscell_5(tck, so[6], l_ba_chip[0], shiftdr, clockdr, so[5]);
idt70v7599_incell bscell_4(tck, so[5], l_ba_chip[1], shiftdr, clockdr, so[4]);
idt70v7599_incell bscell_3(tck, so[4], l_ba_chip[2], shiftdr, clockdr, so[3]);
idt70v7599_incell bscell_2(tck, so[3], l_ba_chip[3], shiftdr, clockdr, so[2]);
idt70v7599_incell bscell_1(tck, so[2], l_ba_chip[4], shiftdr, clockdr, so[1]);
idt70v7599_incell bscell_0(tck, so[1], l_ba_chip[5], shiftdr, clockdr, so[0]);

endmodule

/***********************************************************************************************
#
# MODULE:	tap
# PURPOSE:	jtag tap controller
# DESIGNER:	Tom Diede
# MODIFIED:	Chenhao Geng
#
***********************************************************************************************/

module idt70v7599_tap (jtag_clk_in,
            jtag_rst_in,
            jtag_tms_in,
            jtag_tdi_in,
            jtag_tdo_out,
            jtag_tdo_oe,
            boundary_so,
            is_so,
            shiftdr,
            updatedr,
            clockdr,
            extest,
            highz);   


input          jtag_clk_in;         // jtag clock 
input          jtag_rst_in;         // jtag reset
input          jtag_tms_in;         // jtag mode select
input          jtag_tdi_in;         // jtag data in
output         jtag_tdo_out;        // jtag data out
output         jtag_tdo_oe;         // active high tdo output enable
input          boundary_so;         // shift out from boundary scan reg (pad ring)
input          is_so;               // shift out from internal scan reg 
output         shiftdr;             // shift boundary scan reg
output         updatedr;            // update boundary scan reg
output         clockdr;             // clock the boundary scan reg
output         extest;    // puts boundary scan cells into boundary scan mode
output         highz;        // make all outputs highz

reg          tdo_q;
reg          jt_tdo_premux;
reg          jtag_tdo_out;
reg          jtag_tdo_oe;
reg          extest;
reg          shiftdr;
reg          updatedr;
reg          shiftid;
reg          captureid;
reg          update_q;
reg          shift_q;
reg          capture_q;
reg          shiftby;
reg          captureby;
reg          shiftir;
reg          captureir;
reg          bypass_reg;
reg          capturedr;
reg          highz_q;
reg          highz_q_reg;
reg          highz_latch;
reg  [2:0]   sel_reg_q;          
reg  [2:0]   sel_reg_reg;
reg  [2:0]   sel_reg_latch;
reg          mode_q;
reg          mode_q_reg;
reg  [3:0]   jt_state;           // state register
reg  [3:0]   jt_state_latch;     // clocked on falling edge of jtag_clk
reg  [3:0]   ir_reg;             // instruction register
reg  [3:0]   ir_reg_latch;       // instruction register output latch,
                                 // clocked on falling edge of jtag_clk
reg  [31:0]  jt_id_reg;
reg          id_so;
wire  highz = highz_latch;
wire  tdo_oe_q = (shift_q) ? 1'b1 : 1'b0;
wire  clockdr  = shiftdr || capturedr;

parameter
    test_logic_reset = 4'b1111,
    run_test_idle    = 4'b1100,
    select_dr_scan   = 4'b0111,
    capture_dr       = 4'b0110,
    shift_dr         = 4'b0010,
    exit1_dr         = 4'b0001,
    pause_dr         = 4'b0011,
    exit2_dr         = 4'b0000,
    update_dr        = 4'b0101,
    select_ir_scan   = 4'b0100,
    capture_ir       = 4'b1110,
    shift_ir         = 4'b1010,
    exit1_ir         = 4'b1001,
    pause_ir         = 4'b1011,
    exit2_ir         = 4'b1000,
    update_ir        = 4'b1101;



// id register
   always @(jt_id_reg) id_so = jt_id_reg[0];


// Registers and state bits
always @(posedge jtag_clk_in or posedge jtag_rst_in)
  if (jtag_rst_in)
    begin
      highz_q_reg <= 1'b0;
      sel_reg_reg    <= 3'b001;                   // select device id register
      mode_q_reg     <= 1'b0;
      bypass_reg     <= 1'b0;       
      ir_reg         <= 4'b0010;                  // device id always selected on reset
      jt_id_reg      <= 32'h300067;
      jt_state       <= 4'b1111;                  // test_logic_reset
    end
  else
    begin

      highz_q_reg <= highz_latch;
      sel_reg_reg    <= sel_reg_latch;
      mode_q_reg     <= extest;

      casez (sel_reg_latch)   
        3'b000:                             // bypass register
          begin                   
            ir_reg <= ir_reg_latch;
            if (captureby)    bypass_reg <= 1'b0; // always load 0 during capture bypass
            else if (shiftby) bypass_reg <= jtag_tdi_in;
            else              bypass_reg <= bypass_reg;
          end
        3'b011:                             // instruction register
          begin
            bypass_reg <= bypass_reg;
            if (shiftir)
              begin
                ir_reg[2:0] <= ir_reg_latch[3:1];
                ir_reg[3]   <= jtag_tdi_in;
              end
            else if (captureir)
              begin
                ir_reg <= 4'b0101;          // as per specification
              end
            else
              begin
                ir_reg <= ir_reg_latch;
              end
           end
        default:
           begin
             bypass_reg <= bypass_reg;
             ir_reg     <= ir_reg_latch;
           end
      endcase

// tap state machine 
     casez (jt_state_latch)
       test_logic_reset: 
         jt_state <= jtag_tms_in ? test_logic_reset : run_test_idle;
       run_test_idle:
         jt_state <= jtag_tms_in ? select_dr_scan : run_test_idle;
       select_dr_scan:
         jt_state <= jtag_tms_in ? select_ir_scan : capture_dr;
       capture_dr:
         jt_state <= jtag_tms_in ? exit1_dr : shift_dr;
       shift_dr:
         jt_state <= jtag_tms_in ? exit1_dr : shift_dr;
       exit1_dr:
         jt_state <= jtag_tms_in ? update_dr : pause_dr;
       pause_dr:
         jt_state <= jtag_tms_in ? exit2_dr : pause_dr;
       exit2_dr:
         jt_state <= jtag_tms_in ? update_dr : shift_dr;
       update_dr:
         jt_state <= jtag_tms_in ? select_dr_scan : run_test_idle;
       select_ir_scan:
         jt_state <= jtag_tms_in ? test_logic_reset : capture_ir;
       capture_ir:
         jt_state <= jtag_tms_in ? exit1_ir : shift_ir;
       shift_ir:
         jt_state <= jtag_tms_in ? exit1_ir : shift_ir;
       exit1_ir:
         jt_state <= jtag_tms_in ? update_ir : pause_ir;
       pause_ir:
         jt_state <= jtag_tms_in ? exit2_ir : pause_ir;
       exit2_ir:
         jt_state <= jtag_tms_in ? update_ir : shift_ir;
       update_ir:
         jt_state <= jtag_tms_in ? select_dr_scan : run_test_idle;
       default:
         jt_state <= 4'bxxxx;
     endcase


     // jtag id code
     casez ({captureid, shiftid})
        2'b01: begin                               // shift id
                 jt_id_reg[30:0] <= jt_id_reg[31:1];
                 jt_id_reg[31]   <= jtag_tdi_in;
               end
        2'b10: begin                               // capture id
                     jt_id_reg[31:0] <= 32'h300067;
               end
        default: jt_id_reg <= jt_id_reg;
     endcase
 end





// State decodes
always @(highz_q_reg or sel_reg_reg or bypass_reg or id_so or 
         ir_reg or id_so or jt_state or mode_q_reg)

   begin
         casez (sel_reg_reg)   
           3'b000:                   // bypass register
               tdo_q = bypass_reg;

           3'b001:                   // ID register
               tdo_q = id_so;

           3'b010:                   // boundary scan register 
               tdo_q = 1'b1;

           3'b011:                   // instruction register
               tdo_q = ir_reg[0];

           3'b100:                   // internal scan register
               tdo_q = id_so;

           default:
               tdo_q = bypass_reg;
         endcase


         highz_q = highz_q_reg;
         mode_q     = mode_q_reg;
         sel_reg_q  = sel_reg_reg;

         casez (jt_state) 
             4'b1111:     // test_logic_reset
               begin
                 update_q   = 1'b0; 
                 shift_q    = 1'b0;
                 capture_q  = 1'b0;
                 mode_q     = 1'b0;
                 sel_reg_q  = 3'b001;       // select read-id register
                 highz_q = 1'b0;
               end

             4'b1100:     // run_test_idle  // instructions are executed here
               begin
                 update_q  = 1'b0; 
                 shift_q   = 1'b0;
                 capture_q = 1'b0;
               end

             4'b?110:      // (jt_state == capture_dr || jt_state == capture_ir)
               begin
                 update_q  = 1'b0;
                 shift_q   = 1'b0;
                 capture_q = 1'b1;
               end

             4'b?010:      // (jt_state == shift_dr || jt_state == shift_ir)
               begin
                 update_q  = 1'b0;
                 shift_q   = 1'b1;
                 capture_q = 1'b0;
                 mode_q    = mode_q_reg;
               end
 
             4'b0101:      // (jt_state == update_dr)
               begin
                 update_q  = 1'b1;
                 shift_q   = 1'b0;
                 capture_q = 1'b0;
               end
      
             4'b1101:      // (jt_state == update_ir)
               begin
                 update_q   = 1'b1;
                 shift_q    = 1'b0;
                 capture_q  = 1'b0;
                 highz_q = 1'b0;

                 casez (ir_reg)          // decode instruction reg
                    4'b0000:                          // command = extest 
                      begin
                        sel_reg_q = 3'b010;           // select boundary scan register
                        mode_q    = 1'b1;             // enable boundary scan mode
                      end


                    4'b0001:                          // command = sample/preload
                      begin
                        sel_reg_q = 3'b010;           // select boundary scan register
                        mode_q    = 1'b0;             // de-select boundary scan mode
                      end


                    4'b0010:                          // command = idcode
                        sel_reg_q = 3'b001;           // select id register


                    4'b0100:                          // command = highz
                      begin
                        highz_q = 1'b1;            // highz all outputs
                        mode_q     = 1'b1;            // enable boundary scan mode
                        sel_reg_q  = 3'b000;          // bypass
                      end

                    4'b0011:                          // command = clamp
                      begin
                        sel_reg_q = 3'b000;           // select bypass register
                        mode_q    = 1'b1;             // enable boundary scan mode
                      end

                    4'b0110:                          // command = fuse option programming
                      begin
                        sel_reg_q  = 3'b110;          // select fuse option register
                        mode_q     = 1'b0;            // disable boundary scan mode
                      end

                    4'b1111:                          // command = bypass
                      begin
                        sel_reg_q = 3'b000;           // select bypass register
                        mode_q    = 1'b0;             // disable boundary scan mode
                      end

                    4'b1101: ;                        // command = validate IR,
                                                      // don't need to do anything

                    4'b0101: ;                        // reserved
                    4'b0111: ;                        // reserved
                    4'b1000: ;                        // reserved
                    4'b1001: ;                        // reserved
                    4'b1010: ;                        // reserved
                    4'b1011: ;                        // reserved
                    4'b1100: ;                        // reserved
                    4'b1110: ;                        // reserved

                 endcase
               end
                   
             4'b0100:      // instruction register path select-IR-Scan
               begin
                 update_q  = 1'b0;
                 shift_q   = 1'b0;
                 capture_q = 1'b0;
                 sel_reg_q = 3'b011;                  // select instruction register
               end
   
             4'b0111:     // boundary scan register path - select-DR-scan
               begin
                 update_q  = 1'b0;
                 shift_q   = 1'b0;
                 capture_q = 1'b0;
               end
   
             4'b?001:     // Exit1-DR || Exit1-IR
               begin
                 update_q  = 1'b0;
                 shift_q   = 1'b0;
                 capture_q = 1'b0;
               end
   
             4'b?011:     // Pause-DR || Pause-IR
               begin
                 update_q  = 1'b0;
                 shift_q   = 1'b0;
                 capture_q = 1'b0;
               end
   
             4'b?000:    // Exit2-DR || Exit2-IR
               begin
                 update_q  = 1'b0;
                 shift_q   = 1'b0;
                 capture_q = 1'b0;
               end
         endcase   
   end
   



always @(sel_reg_q or update_q or jt_state)
   begin
      casez (sel_reg_q)
         3'b000: updatedr = 1'b0;                                      // bypass register
         3'b001: updatedr = 1'b0;                                      // read id register
         3'b010: updatedr = (jt_state == update_dr) ? update_q : 1'b0; // boundary scan register
         3'b011: updatedr = 1'b0;                                      // instruction register
         3'b110: updatedr = 1'b0;                                      // internal scan register
         default : updatedr = 1'b0;
      endcase
   end



always @(sel_reg_latch or jt_tdo_premux or boundary_so or is_so)
   begin
      casez (sel_reg_latch)
        3'b000:                  // bypass register
            jtag_tdo_out = jt_tdo_premux;
        3'b001:                  // id register
            jtag_tdo_out = jt_tdo_premux;
        3'b010:                  // boundary scan register
            jtag_tdo_out = boundary_so;
        3'b011:                  // instruction register
            jtag_tdo_out = jt_tdo_premux;
        3'b110:                  // internal scan test
            jtag_tdo_out = is_so;
        default:
            jtag_tdo_out = jt_tdo_premux;
      endcase
   end



  always @(negedge jtag_clk_in or posedge jtag_rst_in)
    if (jtag_rst_in)
      begin
        jt_state_latch   <= 4'b1111;
        ir_reg_latch     <= 4'b0010;     // device id selected on reset
        highz_latch   <= 1'b0;
        sel_reg_latch    <= 3'b001;      // read-id
        extest <= 1'b0;
        jtag_tdo_oe      <= 1'b0;
        jt_tdo_premux    <= 1'b0;
      end
    else
      begin
        jt_state_latch   <= jt_state;
        ir_reg_latch     <= ir_reg;
        highz_latch   <= highz_q;
        sel_reg_latch    <= sel_reg_q;
        extest <= mode_q;
        jtag_tdo_oe      <= tdo_oe_q;
        jt_tdo_premux    <= tdo_q;
      end
   


  always @(negedge jtag_clk_in or posedge jtag_rst_in)
    if (jtag_rst_in)
      begin
        captureir <= 1'b0;
        shiftir   <= 1'b0;
        captureid <= 1'b0;
        shiftid   <= 1'b0;
        capturedr <= 1'b0;
        shiftdr   <= 1'b0;      // module output
        captureby <= 1'b0;
        shiftby   <= 1'b0;
      end
    else
      begin
        casez (sel_reg_q)               	 
          3'b000:			// bypass register
            begin
              captureby <= capture_q;
              shiftby   <= shift_q;
              captureid <= 1'b0;
              shiftid   <= 1'b0;
              capturedr <= 1'b0;
              shiftdr   <= 1'b0;
              captureir <= 1'b0;
              shiftir   <= 1'b0;
            end
          3'b001:			// id register
            begin
              captureby <= 1'b0;
              shiftby   <= 1'b0;
              captureid <= capture_q;
              shiftid   <= shift_q;
              capturedr <= 1'b0;
              shiftdr   <= 1'b0;
              captureir <= 1'b0;
              shiftir   <= 1'b0;
            end
          3'b010: 			// boundary scan register 
            begin
              captureby <= 1'b0;
              shiftby   <= 1'b0;
              captureid <= 1'b0;
              shiftid   <= 1'b0;
              capturedr <= capture_q;
              shiftdr   <= shift_q;
              captureir <= 1'b0;
              shiftir   <= 1'b0;
            end
          3'b011: 			// instruction register
            begin
              captureby <= 1'b0;
              shiftby   <= 1'b0;
              captureid <= 1'b0;
              capturedr <= 1'b0;
              shiftdr   <= 1'b0;
              captureir <= capture_q;
              shiftir   <= shift_q;
            end
          3'b110:                      // internal scan register
            begin
              captureby <= 1'b0;
              shiftby   <= 1'b0;
              captureid <= 1'b0;
              capturedr <= 1'b0;
              shiftdr   <= 1'b0;
              captureir <= 1'b0;
              shiftir   <= 1'b0;
            end
          default :
            begin
              captureby <= 1'bx;
              shiftby   <= 1'bx;
              captureid <= 1'bx;
              shiftid   <= 1'bx;
              capturedr <= 1'bx;
              shiftdr   <= 1'bx;
              captureir <= 1'bx;
              shiftir   <= 1'bx;
            end
        endcase
      end

endmodule

// si - boundary scan serial input     
// so - boundary scan serial output
// di - data from internal logic
// do - data to io block
// pi - data from io block

/************************************************************************
#
# MODULE:	incell
# PURPOSE:	jtag boundary scan input cell
# DESIGNER:	Tom Diede
# MODIFIED:	Chenhao Geng, Shawn Giguere
#
************************************************************************/

module idt70v7599_incell (jtag_clk_in, si, di, shiftdr, clockdr, so);
input   jtag_clk_in;
input   si;
input   di;
input   shiftdr;
input   clockdr;
output  so; 

reg so_internal; 
reg so; 
wire n1 = shiftdr ? si : di;

initial
begin
	so = 0;
	so_internal = 0;
end

always @(posedge jtag_clk_in) so_internal <= clockdr ? n1 : so_internal;
always @(negedge jtag_clk_in) so <= so_internal;
endmodule

/************************************************************************
#
# MODULE:	iocell
# PURPOSE:	jtag boundary scan io cell
# DESIGNER:	Tom Diede
# MODIFIED:	Chenhao Geng, Shawn Giguere
#
************************************************************************/

module idt70v7599_iocell (jtag_clk_in, si, dio, oex, shiftdr, clockdr, updatedr,
                         extest, highz, so, do,jtag_rst_in);
input   jtag_clk_in;
inout   do;
input   oex;
input   si;
inout   dio;
input   shiftdr;
input   clockdr;
input   updatedr;
input   extest;
input	highz;
input	jtag_rst_in;
output  so;

reg scanout1, scanout2, do_reg, do_temp;

wire n1 = shiftdr ? si : do;
wire n2 = shiftdr ? scanout1 : dio;
wire so = scanout2;
wire do = (oex || highz) ? 'hz : do_reg;

initial
begin
    scanout1 <= 0;
    scanout2 <= 0;
    do_reg <= 'hz;
    do_temp <= 0;
end

always @(posedge jtag_clk_in) scanout1 <= jtag_rst_in ? 0 : (clockdr ? n1 : scanout1);
always @(negedge jtag_clk_in) scanout2 <= jtag_rst_in ? 0 : (clockdr ? n2 : scanout2);
always @(negedge jtag_clk_in)
begin
    do_reg <= (jtag_rst_in ? 'hz : ((oex || highz) ? 'hz : (extest ? (updatedr ? scanout2 : do_temp) : 'hz)));
    do_temp <= updatedr ? scanout2 : do_temp;
    scanout2 <= jtag_rst_in ? 0 : (clockdr ? n2 : scanout2);
end
always @(posedge jtag_rst_in)
begin
    scanout1 <= 0;
    scanout2 <= 0;
    do_reg <= (jtag_rst_in ? 'hz : ((oex || highz) ? 'hz : (extest ? 0 : 'hz)));
    do_temp <= 0;
end
endmodule

/***********************************************************************
#
# MODULE:	outcell
# PURPOSE:	jtag boundary scan output cell
# DESIGNER:	Tom Diede
# MODIFIED:	Chenhao Geng, Shawn Giguere
#
***********************************************************************/

module idt70v7599_outcell (jtag_clk_in, si, di, shiftdr, clockdr, updatedr,
                          extest, highz, so, do);
input   jtag_clk_in;
input   si;
input   di;
input   shiftdr;
input   clockdr;
input   updatedr;
input   extest;
input	highz;
output  so;
output  do; 

reg so_internal;
reg so;
reg n2;
wire n1 = shiftdr ? si : di;
wire do = highz ? 1'bz : (extest ? n2 : di);

initial
begin
	n2 = 0;
	so = 0;
	so_internal = 0;
end

always @(posedge jtag_clk_in) so_internal <= clockdr ? n1 : so_internal;
always @(negedge jtag_clk_in)
begin
    n2 <= updatedr ? so : n2;
    so <= so_internal;
end
endmodule

/**********************************************************************
#
# MODULE:	oecell
# PURPOSE:	jtag boundary scan output enable control cell
# DESIGNER:	Tom Diede
# MODIFIED:	Chenhao Geng, Shawn Giguere
#
**********************************************************************/

module idt70v7599_oecell (jtag_clk_in, si, di, shiftdr, clockdr, updatedr,
                         extest, so, do);
input   jtag_clk_in;
input   si;
input   di;
input   shiftdr;
input   clockdr;
input   updatedr;
input   extest;
output  so;
output  do; 

reg so_internal;
reg so;
reg n2;

initial
begin
	n2 = 0;
	so = 0;
	so_internal = 0;
end

wire n1 = shiftdr ? si : di;
wire do = extest ? n2 : di;

always @(posedge jtag_clk_in)
begin
    so_internal <= clockdr ? n1 : so_internal;
end

always @(negedge jtag_clk_in)
begin
    n2 <= updatedr ? so : n2;
    so <= so_internal;
end
endmodule
