---------------------------------------------------------------------
-- idt70v7339
--      Entities and Architectures used in idt70v7339 or related parts
--              are defined and described first.
--
-- HISTORY:
-- 08/24/01 xyz  made some changes on input signals registering issues
--               1). the rw signals are single registered now
--               2). the be signals are now handled in the same way as 
--                   the ce signals.
--               3). the clocks used in the MEM module are delayed 1ns
--                   to avoid potential racing conditions, therefore,
--                   all the delay timings used in the MEM module are
--                   reduced by 1ns.
-- 01/11/01 XYZ  made many changes in IO and MEM module.
-- 06/27/00 SMG  Initial version: Adapted from idt70v3579 VHDL model
-------------------------------------------------------------------
--   Copyright 1999, Integrated Device Technology, Inc.
--   All Rights Reserved
-------------------------------------------------------------------
--* Integrated Device Technology, Inc. ("IDT") hereby grants the
-- user of this VHDL model a non-exclusive, nontransferable 
-- license to use this VHDL model under the following terms.   
-- The user is granted this license only to use the VHDL    
-- model and is not granted rights to sell, copy (except as needed   
-- to run the IBIS model), rent, lease or sub-license the VHDL   
-- model in whole or in part, or in modified form to anyone. The User   
-- may modify the VHDL model to suit its specific applications,   
-- but rights to derivative works and such modifications shall belong   
-- to IDT. * 
--*  This VHDL model is provided on an "AS IS" basis and
-- IDT makes absolutely no warranty with respect to the information
-- contained herein. IDT DISCLAIMS AND CUSTOMER WAIVES ALL
-- WARRANTIES, EXPRESS AND IMPLIED, INCLUDING WARRANTIES OF
-- MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
-- ENTIRE RISK AS TO QUALITY AND PERFORMANCE IS WITH THE
-- USER ACCORDINGLY, IN NO EVENT SHALL IDT BE LIABLE 
-- FOR ANY DIRECT OR INDIRECT DAMAGES, WHETHER IN CONTRACT OR
-- TORT, INCLUDING ANY LOST PROFITS OR OTHER INCIDENTAL,
-- CONSEQUENTIAL, EXEMPLARY, OR PUNITIVE DAMAGES ARISING OUT OF
-- THE USE OR APPLICATION OF THE VERILOG/VCS model.  Further, 
-- IDT reserves the right to make changes without notice to any 
-- product herein to improve reliability, function, or design.   
-- IDT does not convey any license under patent rights or 
-- any other intellectual property rights, including those of   
-- third parties. IDT is not obligated to provide maintenance   
-- or support for the licensed VHDL model. *
---------------------------------------------------------------



library IEEE;
use IEEE.std_logic_1164.all;

entity idt70v7339 is
  -- parameters for 166Mhz part
  -- The numbers can be easily changed for other speed grades
  generic (
        partid :string    := "idt70v7339";
        tsetup :time      := 1.7 ns;
        thold  :time      := 0.5 ns;
        tcd    :time      := 3.6 ns;
        toe    :time      := 3.6 ns;
        tckhz  :time      := 3 ns;
        tohz   :time      := 3.5 ns;
        addr_bit :integer := 19;
        width    :integer := 18);
  port (
    clkl, clkr	: in    std_logic;  -- clock input
    rwxl, rwxr	: in    std_logic;  -- read/write (low)
    ce0lx, ce1l	: in    std_logic;  -- chip enables, 0 is active low
    ce0rx, ce1r	: in    std_logic;  -- 1 is active high
    ublx, lblx	: in    std_logic; 	-- byte enable, active low
		ubrx, lbrx	:	in		std_logic;	-- byte enable, active low
    oelx, oerx	: in    std_logic;  -- output enable, active low
    l_a, r_a		: in    std_logic_vector (addr_bit-7 downto 0); -- input address
    l_ba, r_ba	: in    std_logic_vector (5 downto 0); -- input address
    adslx, adsrx	: in    std_logic;  -- address strobe
    repeatlx, repeatrx	: in    std_logic;  -- counter repeat, active low
    pipeftlx, pipeftrx	: in    std_logic;  -- pipeline/flow through (low)
    cntenlx, cntenrx	: in	std_logic;  -- count enable, active low
    iol, ior		: inout	std_logic_vector (width-1 downto 0)); -- ios
end idt70v7339;

-------------------------------------------------------------------
-- ENTITY:       io
-- PURPOSE:      data io and input register
-----------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
USE IEEE.vital_timing.all;
LIBRARY STD;
USE STD.TEXTIO.ALL;

ENTITY io_idt70v7339 is
	generic (
		instancepath	: string:=" ";
		partID	: string := "io_idt70v7339";
		tsetup	: time:=1.7 ns; 
		thold		: time:=0.5 ns; 
		tcd			: time:= 3.6 ns;
		toe			: time:= 3.6 ns;
		tckhz		: time:= 3 ns;
		tohz		: time:= 3.5 ns;
		addr_bit	: integer:=19;
		width			: integer:=18);
	port(
		ck,rwx,ce0x,ce1,oex :	in	std_logic;
		cebuf, rwbuf :		inout	std_logic;
		ubx, lbx :			in 	std_logic;
		ubbuf,lbbuf :			inout	std_logic;
		data :			inout	std_logic_vector(width-1 downto 0); 
		ce2buf:			inout	std_logic;
		pfx :			in	std_logic;
		io :			inout	std_logic_vector(width-1 downto 0) );
END io_idt70v7339;

-------------------------------------------------------------------
-- ENTITY:       memory
-- PURPOSE:      memory core
-----------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
USE IEEE.vital_timing.all;
LIBRARY STD;
USE STD.TEXTIO.ALL;

ENTITY memory_idt70v7339 is
	generic (
		partID : string 	:= "memory_idt70v7339";
		tcd : time := 2.6 ns;  -- value reduced by 1ns because clocks are
		tft : time := 9 ns;    -- delayed 1ns.
		width : integer 	:= 18;
		addr_bit: integer	:= 19
		);
	port(
		lmck, rmck	: in	std_logic;
		lrwx, rrwx	: in	std_logic;
		loex, roex	: in 	std_logic;
		lce, rce		: in	std_logic;
		l2ce, r2ce	: in	std_logic;
		lpfx,rpfx	:	in	std_logic;
		ubl, lbl	: in	std_logic;
		ubr, lbr	: in	std_logic;
		laddr, raddr	: in	std_logic_vector(addr_bit-1 downto 0);
		ldata, rdata	: inout	std_logic_vector(width-1 downto 0)
		);
	subtype word is std_logic_vector (width -1 downto 0);
	constant nword : integer := 2 ** addr_bit;
   type ramtype is array(0 to nword-1) of word;
END memory_idt70v7339;

-------------------------------------------------------------------
-- ENTITY:       counter/address register
-- PURPOSE:      A 14 to 16 bit counter (depending on the part)
--                       and address register
-----------------------------------------------------------------
library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
USE IEEE.vital_timing.all;
LIBRARY STD;
USE STD.TEXTIO.ALL;

ENTITY counter_idt70v7339 is
	generic (
      InstancePath : string := " ";
      partID : string := "counter_idt707519";
		tsetup: time:=1.7 ns; 
		thold: time:=0.5 ns; 
		addr_bit: integer:=19
		);
	port(
		clk		: in	std_logic;
		a_in	: in	std_logic_vector(addr_bit-7 downto 0);
		ba_in	: in	std_logic_vector(5 downto 0);
		aout	: out	std_logic_vector(addr_bit-1 downto 0):=(others=>'0');
		repeatx	: in	std_logic;
		adsx		: in	std_logic;
		cntenx	: in	std_logic
    );
END counter_idt70v7339;

-------------------------------------------------------------------
-- ENTITY:       complete chip
-- PURPOSE:      hook together sub-modules
-----------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
USE IEEE.vital_timing.all;
LIBRARY STD;
USE STD.TEXTIO.ALL;

entity chip_idt70v7339 is
   generic (
      partid : string;
      tsetup, thold, tcd, toe, tckhz, tohz : time;
      addr_bit: integer:=19;
      width: integer:=18
      );
	port (
		lck, rck,	
		lrwx, rrwx,	
		lce0x, lce1,
		rce0x, rce1,
		loex, roex,		
		ladsx, radsx, 	
 		lpfx, rpfx, 	
 		lrepeatx, rrepeatx, 	
		lcntenx, rcntenx : in std_logic; 
		ublx, lblx: 	in 	std_logic;		
		ubrx, lbrx:		in	std_logic;
		l_a, r_a:	in std_logic_vector(addr_bit-7 downto 0);  
		l_ba, r_ba:	in std_logic_vector(5 downto 0);  
		lio, rio:	inout std_logic_vector(width-1 downto 0) );				
end chip_idt70v7339;

-------------------------------------------------------------------
-- ARCHITECTURE:  complete chip hookup
-- PURPOSE:      data io and input register
-----------------------------------------------------------------
architecture structure of chip_idt70v7339 is

component counter_idt70v7339
generic (
  instancepath : string;
  partid       : string;
  thold, tsetup: time;
  addr_bit:integer);
port (
      clk			: in  std_logic;
      a_in		: in  std_logic_vector(addr_bit-7 downto 0);
      ba_in		: in  std_logic_vector(5 downto 0);
      aout		: out std_logic_vector(addr_bit-1 downto 0);
      repeatx, adsx, cntenx	: in  std_logic);
end component;

component io_idt70v7339 is
    generic (
      instancepath,partid :string;
      tsetup,thold,tcd,toe,tckhz,tohz:time;
      addr_bit,width:integer);
    port(
      ck,rwx,ce0x,ce1,oex,pfx	: in  std_logic;
      cebuf, ce2buf, rwbuf	: inout std_logic;
      ubx, lbx	: in  std_logic;
      ubbuf, lbbuf	: inout  std_logic;
      data	: inout  std_logic_vector(width-1 downto 0);
      io		: inout  std_logic_vector(width-1 downto 0) );
END component;

component memory_idt70v7339 is
   generic (
	   tcd 		: time;
     partid :string;
     addr_bit,width : integer);
   port(
      lmck, rmck	: in  std_logic;
      lrwx, rrwx	: in  std_logic;
			loex, roex	:	in	std_logic;
      lce, rce		: in  std_logic;
      l2ce, r2ce	: in  std_logic;
			lpfx, rpfx	: in 	std_logic;
      ubl, lbl		: in  std_logic;
			ubr, lbr		:	in	std_logic;
      laddr, raddr	: in  std_logic_vector(addr_bit-1 downto 0);
      ldata, rdata	: inout std_logic_vector(width-1 downto 0)
    );
END component;

signal laddr_i, raddr_i : std_logic_vector(addr_bit-1 downto 0);
signal lce_i, rce_i, l2ce_i, r2ce_i, lrw_i, rrw_i : std_logic;
signal ubl_i, lbl_i, ubr_i, lbr_i : std_logic;
signal ldata, rdata : std_logic_vector(width-1 downto 0):=(others=>'Z');
signal lmck, rmck : std_logic ;   

begin

-- lmck and rmck are the delayed clocks used in the memory core.
lmck <= lck after 1 ns;
rmck <= rck after 1 ns;
	
lcounter : counter_idt70v7339
  generic map (
    instancepath => "Left",
    partid       => partid,
    tsetup       => tsetup,
    thold        => thold,
    addr_bit     => addr_bit)
	port map(clk=>lck, a_in=>l_a, ba_in=>l_ba, aout=>laddr_i,
		 repeatx=>lrepeatx, adsx=>ladsx, cntenx=>lcntenx);
rcounter : counter_idt70v7339
  generic map (
    instancepath => "Right",
    partid       => partid,
    tsetup       => tsetup,
    thold        => thold,
    addr_bit     => addr_bit)
	port map(clk=>rck, a_in=>r_a, ba_in=>r_ba, aout=>raddr_i,
		 repeatx=>rrepeatx, adsx=>radsx, cntenx=>rcntenx);

lio1 : io_idt70v7339
  generic map (
    instancepath => "Left",
    partid       => partID,
    tsetup       => tsetup,
    thold        => thold,
    tcd          => tcd,
    toe          => toe,
    tckhz        => tckhz,
    tohz         => tohz,
    addr_bit     => addr_bit,
    width        => width)
	port map(ck=>lck, rwx=>lrwx, ce0x=>lce0x, ce1=>lce1, oex=>loex, 
		 cebuf=>lce_i, ce2buf=>l2ce_i, rwbuf=>lrw_i, ubx=>ublx, lbx=>lblx,
		 ubbuf=>ubl_i, lbbuf=>lbl_i, pfx=>lpfx, data=>ldata, io=>lio);

rio1 : io_idt70v7339
  generic map (
    instancepath => "Right",
    partid       => partID,
    tsetup       => tsetup,
    thold        => thold,
    tcd          => tcd,
    toe          => toe,
    tckhz        => tckhz,
    tohz         => tohz,
    addr_bit     => addr_bit,
    width        => width)
	port map(ck=>rck, rwx=>rrwx, ce0x=>rce0x, ce1=>rce1, oex=>roex, 
		 cebuf=>rce_i, ce2buf=>r2ce_i, rwbuf=>rrw_i, ubx=>ubrx, lbx=>lbrx,
		 ubbuf=>ubr_i, lbbuf=>lbr_i, pfx=>rpfx, data=>rdata, io=>rio);

mcore : memory_idt70v7339
  generic map (
    partid   => partID,
		tcd => tcd,
    width    => width,
    addr_bit => addr_bit)
	port map(lmck=>lmck, rmck=>rmck, lrwx=>lrw_i, rrwx=>rrw_i, ubl=>ubl_i,
		 lbl=>lbl_i, ubr=>ubr_i, lbr=>lbr_i, 
		 lce=>lce_i, rce=>rce_i, l2ce=>l2ce_i,
		 loex => loex, roex => roex,
		 r2ce=>r2ce_i, laddr=>laddr_i, raddr=>raddr_i, ldata=>ldata, 
		 rdata=>rdata, lpfx => lpfx, rpfx => rpfx);

assert not(laddr_i = raddr_i and lrw_i = '0' and rrw_i = '0' and
is_x(laddr_i)) report
  "Write Error, both ports writing to same address" severity warning;

end structure;

-------------------------------------------------------------------
-- ARCHITECTURE:       io
-- PURPOSE:      data io and input register
-----------------------------------------------------------------
ARCHITECTURE behavioral of io_idt70v7339 is
signal oeio		: std_logic;
signal inreg	: std_logic_vector(width-1 downto 0);
signal outreg, datareg, io_i	: std_logic_vector(width-1 downto 0);
signal enio, beck	: std_logic_vector(1 downto 0);

begin

-- incorporate tcd and thz into control signals
oeio <= '1' after tohz when oex = '0'
				else '0' after toe;
enio(1) <= '1' after tcd when (rwbuf and ce2buf and beck(1)) = '1'
				else '0' after tckhz;
enio(0) <= '1' after tcd when (rwbuf and ce2buf and beck(0)) = '1'
				else '0' after tckhz;

io_i(8 downto 0) <= outreg(8 downto 0) when (oeio and enio(0)) = '1'
						else	(others=>'Z');
io_i(17 downto 9) <= outreg(17 downto 9) when (oeio and enio(1)) = '1'
						else	(others=>'Z');

io <= io_i(width-1 downto 0);

data <= inreg when (rwbuf = '0' and cebuf /= 'X')
					else (others=>'Z');

second:	process (ck, cebuf)
-- the first process incorporates setup/hold errors
-- this process is simply the second stage pipe

	begin
		if pfx = '0' then
			ce2buf <= cebuf;
			beck(0) <= lbbuf;
			beck(1) <= ubbuf;
		else
			if rising_edge(ck) then
				ce2buf <= cebuf;
				beck(0) <= lbbuf;
				beck(1) <= ubbuf;
			end if;
		end if;
	end process;

output: process (data,ck)
	begin
	  if oex = '0' then
		if (pfx) = '0' then
			if data'EVENT then
				outreg(width-1 downto 0) <= data;
			end if;
		else
				outreg(width-1 downto 0) <= data;
		end if;
		end if;
	end process;					

-- perform setup/hold checks, and 1st stage FFs
timing_checks: process(ck, io, ce0x, ce1, rwx, lbx, ubx)
  variable io_time, ce0_time,ce1_time, rw_time, lb_time, ub_time
                  : VitalTimingDataType;
  variable io_viol, ce0_viol, ce1_viol, rw_viol, lb_viol, ub_viol
                  : X01 := '0';
  variable violation : x01 := '0';
  begin
  -- io check
       VitalSetupHoldCheck (            
          TestSignal      => io,
          TestSignalName  => "IO",
          RefSignal       => CK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => io_time,
          Violation       => io_viol );
--  ce check
       VitalSetupHoldCheck (            
          TestSignal      => ce0x,
          TestSignalName  => "CE0_",
          RefSignal       => CK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => ce0_time,
          Violation       => ce0_viol );
       VitalSetupHoldCheck (            
          TestSignal      => ce1,
          TestSignalName  => "CE1",
          RefSignal       => CK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => ce1_time,
          Violation       => ce1_viol );
  -- rwx check
       VitalSetupHoldCheck (            
          TestSignal      => rwx,
          TestSignalName  => "RW_",
          RefSignal       => CK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => rw_time,
          Violation       => rw_viol );
  -- be check
       VitalSetupHoldCheck (            
          TestSignal      => ubx,
          TestSignalName  => "UB_",
          RefSignal       => CK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => ub_time,
          Violation       => ub_viol );
       VitalSetupHoldCheck (            
          TestSignal      => lbx,
          TestSignalName  => "LB_",
          RefSignal       => CK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => lb_time,
          Violation       => lb_viol );

      violation:= ub_viol or lb_viol or rw_viol
                  or ce0_viol or ce1_viol or io_viol;

-- now due first stage pipe
			if rising_edge(ck) then
        if ce0_viol = '0' and ce1_viol = '0' then
          cebuf <= not(ce0x) and ce1;
        else
            cebuf <= 'X';
        end if;
        if rw_viol = '0' then
          rwbuf <= rwx;
        else
          rwbuf <= 'X';
        end if;
        if ub_viol = '0'  then
          ubbuf <= not (ubx);
        else
          ubbuf <= 'X';
        end if;
        if lb_viol = '0'  then
          lbbuf <= not (lbx);
        else
          lbbuf <= 'X';
        end if;
			end if;
        if io_viol = '0'  then
          inreg <= io;
        else
          inreg <= (others => 'X');
        end if;
		
end process;
  
end behavioral;

-------------------------------------------------------------------
-- ARCHITECTURE:       memory
-- PURPOSE:      memory core
-----------------------------------------------------------------

ARCHITECTURE behavioral of memory_idt70v7339 is

shared variable mem: ramtype;
--	signal lrbuf: std_logic_vector(width-1 downto 0);
--	signal lwbuf: std_logic_vector(width-1 downto 0);

begin
-- left port memory access
lwrite : process(lmck,lrwx)
	variable addr:integer;
	variable lrbuf: std_logic_vector(width-1 downto 0);
	variable lwbuf: std_logic_vector(width-1 downto 0);
	variable raddr_temp: std_logic_vector(addr_bit-1 downto 0);
	variable laddr_temp: std_logic_vector(addr_bit-1 downto 0);
  
   begin
	 if lrwx = '0' then
	 		ldata <= (others => 'Z');
		end if;
	
	if rce = '1' then
		raddr_temp := raddr;
	else
		raddr_temp := (others => 'X');
	end if;
	
	if lce = '1' then
		laddr_temp := laddr;
	else
		laddr_temp := (others => 'X');
	end if;
	
-- code made longer to account for signals going into
		 if not(is_x(laddr_temp)) then
       -- only do access if valid address and chip is enabled
			addr := to_integer(unsigned(laddr));	 
      if rising_edge(lmck) then
         if lrwx='1' then -- read operation
					 	if lpfx = '1' then
             ldata <= lrbuf after tcd;
						else 
							if laddr_temp(18 downto 13) = raddr_temp(18 downto 13) then
								ldata <= (others => 'X') after tft;
							else
								ldata <= mem(addr) after tft;
							end if;
						end if;
						if laddr_temp(18 downto 13) = raddr_temp(18 downto 13) then
							ldata <= (others => 'X');
						else
	           lrbuf := mem(addr);
						end if;
						
         else -- write operation
         -- do this if rw=0 or x, if rw=x then ldata = z
           ldata<= (others=>'Z');
           if laddr_temp(18 downto 13) = raddr_temp(18 downto 13) then
             -- error, both ports accessing the same address
             lwbuf := (others => 'X');
           else
             if lbl = '1' then
               lwbuf(8 downto 0) := ldata(8 downto 0);
             elsif lbl = 'X' then
               lwbuf(8 downto 0) := (others => 'X');
             end if;
             if ubl = '1' then
               lwbuf(17 downto 9) := ldata(17 downto 9);
             elsif ubl = 'X' then
               lwbuf(17 downto 9) := (others => 'X');
             end if;
           mem(addr) := lwbuf;
					 end if;
       end if;
     end if;
   end if;
end process;

-- right port memory access
rwrite : process(rmck,rrwx)
	variable addr:integer;
	variable rrbuf: std_logic_vector(width-1 downto 0);
	variable rwbuf: std_logic_vector(width-1 downto 0);
	variable laddr_temp: std_logic_vector(addr_bit-1 downto 0);
	variable raddr_temp: std_logic_vector(addr_bit-1 downto 0);

   begin
     -- code made longer to account for signals going into
     -- X and corrupting the data
		 if (rrwx = '0') then
			rdata <= (others => 'Z');
		 end if;
		
		if lce = '1' then
			laddr_temp := laddr;
		else
			laddr_temp := (others => 'X');
		end if;
		
		if rce = '1' then
			raddr_temp := raddr;
		else
			raddr_temp := (others => 'X');
		end if;
		
     if not(is_x(raddr_temp)) then
       -- only do access if valid address and chip is enable
			 addr := to_integer(unsigned(raddr));
      if rising_edge(rmck) then
         if rrwx='1' then -- read operation
					 	if rpfx = '1' then
      				rdata <= rrbuf after tcd;
						else
							if laddr_temp(18 downto 13) = raddr_temp(18 downto 13) then
								rdata <= (others => 'X') after tft;
							else
								rdata <= mem(addr) after tft;
							end if;
						end if;
						if laddr_temp(18 downto 13) = raddr_temp(18 downto 13) then
							rrbuf := (others => 'X');
						else
	           rrbuf := mem(addr);
						end if;
						
         else -- write operation
           rdata<= (others=>'Z');
           rwbuf := mem(addr);
           if laddr_temp(18 downto 13) = raddr_temp(18 downto 13) then
             -- error, both ports accessing the same address
             rrbuf := (others => 'X');
             rwbuf := (others => 'X');
           else
             if lbr = '1' then
               rwbuf(8 downto 0) := rdata(8 downto 0);
             elsif lbr = 'X' then
               rwbuf(8 downto 0) := (others => 'X');
             end if;
             if ubr = '1' then
               rwbuf(17 downto 9) := rdata(17 downto 9);
             elsif ubr = 'X' then
               rwbuf(17 downto 9) := (others => 'X');
             end if;
				    mem(addr) := rwbuf;
					end if;
       end if;
     end if;
   end if;
end process;

end behavioral;

-------------------------------------------------------------------
-- ARCHITECTURE:       counter/address register
-- PURPOSE:      A 14 to 16 bit counter (depending on the part)
--                       and address register
-----------------------------------------------------------------

ARCHITECTURE behavioral of counter_idt70v7339 is
  signal ain		: std_logic_vector(addr_bit-1 downto 0);
  signal violation	: X01 := '0';
begin
	process (clk, violation)
		variable cntrl : std_logic_vector (2 downto 0);
		variable count : unsigned (addr_bit-1 downto 0):=('0','0',others=>'1');
		variable repeat_adr : unsigned (addr_bit-1 downto 0):=('0','0',others=>'1');
	begin
		ain(addr_bit-1 downto addr_bit-6) <= ba_in;
		ain(addr_bit-7 downto 0) <= a_in;
		if rising_edge(clk) and violation='0' then
			cntrl := cntenx & adsx & repeatx;
			case cntrl is
				when "000" => count := repeat_adr; -- load repeat_adr
				when "010" => count := repeat_adr; -- load repeat_adr
				when "100" => count := repeat_adr; -- load repeat_adr
				when "110" => count := repeat_adr; -- load repeat_adr
				when "101" => count := unsigned(ain); repeat_adr := unsigned(ain);
				when "001" => count := unsigned(ain); repeat_adr := unsigned(ain);
				when "011" => count := (count + 1) mod (2**addr_bit);
				when others => count := count;
			end case;
		end if;
      if violation /='0'  then
        count:=(others => 'X');
      end if;
		aout <= std_logic_vector(count);
	end process;

timing_checks: process(clk, repeatx, cntenx, adsx, ain)
  variable rpt_time,cnten_time,ads_time,addr_time  : VitalTimingDataType;
  variable rpt_viol,cnten_viol,ads_viol,addr_viol  : X01 := '0';
  
  begin
  -- reset check
       VitalSetupHoldCheck (            
          TestSignal      => repeatx,
          TestSignalName  => "CNTRPT_",
          RefSignal       => CLK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => rpt_time,
          Violation       => rpt_viol );
  -- adsx check
       VitalSetupHoldCheck (            
          TestSignal      => adsx,
          TestSignalName  => "ADS_",
          RefSignal       => CLK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => ads_time,
          Violation       => ads_viol );
  -- cntenx check
       VitalSetupHoldCheck (            
          TestSignal      => cntenx,
          TestSignalName  => "CNTEN_",
          RefSignal       => CLK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => cnten_time,
          Violation       => cnten_viol );
  -- address check
       VitalSetupHoldCheck (            
          TestSignal      => ain,
          TestSignalName  => "A",
          RefSignal       => CLK,
          RefSignalName   => "CLK",
          SetupHigh       => tsetup,
          SetupLow        => tsetup,
          HoldHigh        => thold,
          HoldLow         => thold,
          RefTransition   => '/',
          HeaderMsg       => partID & InstancePath,
          TimingData      => addr_time,
          Violation       => addr_viol );
   violation <= addr_viol or cnten_viol or ads_viol or rpt_viol;
  end process;
  
end behavioral;

-------------------------------------------------------------------
--
-- idt70v7339
--
--      Now do part specific definitions and calls
--
-------------------------------------------------------------------
architecture behavioral of idt70v7339 is
  component chip_idt70v7339
	port (
		lck, rck			: in std_logic; 
		lrwx, rrwx		: in std_logic; 	
		lce0x, lce1		: in std_logic; 
		rce0x, rce1		: in std_logic; 
		loex, roex		: in std_logic; 		
		ladsx, radsx	: in std_logic; 	
		lpfx, rpfx		: in std_logic; 
 		lrepeatx, rrepeatx	: in std_logic; 	
		lcntenx, rcntenx		: in std_logic; 
		lblx, ublx		: in std_logic;		
		lbrx, ubrx		:	in std_logic;
		l_a, r_a			: in std_logic_vector(addr_bit-7 downto 0);  
		l_ba, r_ba		: in std_logic_vector(5 downto 0);  
		lio, rio			: inout std_logic_vector(width-1 downto 0) );				
  end component;

  for x0 : chip_idt70v7339
    use entity work.chip_idt70v7339(structure)
      generic map (
        partid       => partid,
        tsetup       => tsetup,
        thold        => thold,
        tcd          => tcd,
        toe          => toe,
        tckhz        => tckhz,
        tohz         => tohz,
        addr_bit     => addr_bit,
        width        => width);
  
begin  -- behavioral

x0:chip_idt70v7339

port map (
  lck     => clkl, 
  rck     => clkr, 
  lrwx    => rwxl, 
  rrwx    => rwxr, 
  lce0x   => ce0lx,
  lce1    => ce1l,
  rce0x   => ce0rx,
  rce1    => ce1r,
  loex    => oelx,
  roex    => oerx,
  lpfx    => pipeftlx,
  rpfx    => pipeftrx,
  ladsx   => adslx,
  radsx   => adsrx,
  lrepeatx   => repeatlx,
  rrepeatx   => repeatrx,
  lcntenx => cntenlx,
  rcntenx => cntenrx,
  lblx    => lblx,
	ublx		=> ublx,
	lbrx		=> lbrx,
	ubrx		=> ubrx,
  l_a     => l_a,
  r_a     => r_a,
  l_ba    => l_ba,
  r_ba    => r_ba,
  lio     => iol,
  rio     => ior);

end behavioral;


